﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Globalization;
using System.Windows.Forms;
using System.IO;

using System.Threading;
using System.Threading.Tasks;

using Ivi.Driver.Interop;
using Agilent.AgM9393.Interop;
using Agilent.AgM938x.Interop;
using Agilent.AgM90XA.Interop;
using Agilent.M9000;
using Agilent.M9000.Interfaces;
using Agilent.AgM9018.Interop;
using Ivi.ConfigServer.Interop;

namespace NSM9393A
{
    public partial class Vsag_M9393A
    {
        // Instrument control Methods

        #region Init and Close Instruments
        public void initVsgVsa()
        {
            int errorcode = -1; string message = string.Empty;
            bool idquery = true;
            bool reset = true;
            string M9393options = string.Format(
                "QueryInstrStatus=true, Simulate={0}, M9393Setup= Model=, Trace=false, ShareAllVisaSessions=true", (simulateHardware ? "true" : "false"));
            string M9381options = string.Format(
                "QueryInstrStatus=true, Simulate={0},  ShareAllVisaSessions=true, UsePlayArbThreadPool=true", (simulateHardware ? "true" : "false"));
            try
            {
                Vsg = new AgM938x();

                // Start a task to initialize VSG
                Task vsg = Task.Factory.StartNew(() =>
                    {
                        //Initialize M9381
                        Vsg.Initialize(VsgResource, idquery, reset, M9381options);
                        // Clear startup messages & warnings if any.
                        do
                        {
                            Vsg.Utility.ErrorQuery(ref errorcode, ref message);
                            if (errorcode != 0)
                                Console.WriteLine(message);
                        } while (errorcode != 0);

                        // Adjust PLL for best performance for WLAN, GSM and EDGE
                        Vsg.Modules.Synthesizer.PLLMode = AgM938xSynthesizerPLLModeEnum.AgM938xSynthesizerPLLModeBestWideOffset;
                        //Do not use ALC with BasebandPower command
                        Vsg.ALC.Enabled = false;
                        Vsg.RF.OutputEnabled = true;
                        Vsg.Modulation.Enabled = true;
                        Vsg.System.GcTimingOptimizationEnabled = false;
                    }
                    );
                //Initialize the M9300 M-App software, if required for this device
                if (useM9300)
                {
                    // Check to see if the M9391Resource is a valid IVI Configuration Store Alias
                    var store = new IviConfigStore();
                    store.Deserialize(store.MasterLocation);
                    bool validAlias = false;
                    string IviSoftwareModuleName = ResourcePluginType.Replace(".Ivi", "");
                    foreach (IviDriverSession driverSession in store.DriverSessions)
                    {
                        if (driverSession.SoftwareModuleName.Equals(IviSoftwareModuleName,
                                                                    StringComparison.CurrentCultureIgnoreCase) &&
                            driverSession.Name.Equals(VsaResource, StringComparison.CurrentCultureIgnoreCase))
                        {
                            validAlias = true;
                            break;
                        }
                    }
                    if (!validAlias)
                    {
                        Exception ex = new Exception("M9393A Resource must be IVI Configuration Store Alias\n" +
                            "Use M9393A SFP to create Saved Connection named " + VsaResource);
                        throw (ex);
                    }

                    // Initialize the M9300 Resource Manager
                    M9000 = SessionFactory.OpenM9000Session();

                    // Create sharable IVI resource 
                    var result = M9000.OpenSharableResource(ResourcePluginType, VsaResource, out _key, simulateHardware);
                    if (result != ResourceManagerResult.Ok)
                    {
                        Exception ex = new Exception("Could not create sharable IVI driver session.");
                        throw (ex);
                    }
                }
                else
                {
                    Vsa = new AgM9393();
                    //Initialize M9393
                    Vsa.Initialize(VsaResource, idquery, reset, M9393options);
                    // Clear startup messages & warnings if any.
                    do
                    {
                        Vsa.Utility.ErrorQuery(ref errorcode, ref message);
                        if (errorcode != 0)
                            Console.WriteLine(message);
                    } while (errorcode != 0);
                    // Adjust PLL for best performance for WLAN, GSM and EDGE
                    //Vsa.Modules.Synthesizer.PLLMode = AgM9393SynthesizerPLLModeEnum.AgM9393SynthesizerPLLModeBestWideOffset;
                }
                // Wait for VSG to complete
                vsg.Wait();
            }
            catch (Exception ex)
            {
                throw (ex);
            }
        }

        public void closeVsgVsa()
        {
            try
            {
                if (Vsg != null)
                    if (Vsg.Initialized)
                    {
                        Vsg.Close();
                        Vsg = null;
                    }

                // Close M90XA if it is running
                if (M90XA != null)
                {
                    AgVisa32.viClose(xApp);
                    unlockVsa();
                    M90XA.Close();
                    M90XA = null;
                }

                if (M9000 != null)
                {
                    M9000.CloseM9000Session();
                    M9000 = null;
                }
                if (Vsa != null)
                {
                    if (Vsa.Initialized)
                    {
                        Vsa.Close();
                        Vsa = null;
                    }
                }
            }
            catch (Exception ex)
            {
                throw (ex);
            }
        }

        public void configureTriggers()
        {
            // Create a connection to the Chassis
            bool idquery = true;
            bool reset = true;
            string M9018options = string.Format(
                "QueryInstrStatus=true, Simulate={0}, M9391Setup= Model=, Trace=false", (simulateHardware ? "true" : "false"));
            try
            {
                IAgM9018 M9018 = new AgM9018();
                //Initialize M9018A
                M9018.Initialize(M9018resource, idquery, reset, M9018options);

                // Configure the Chassis to allow Trigger Segment 1 to driver trigger segment 2 for PXI Trigger2.
                // Use PXI Trig2 as it is connected to the M90XA as "EXT2"
                M9018.TriggerBus.Connect(2, AgM9018TrigBusEnum.AgM9018TrigBus1To2To3);
                M9018.Close();
                M9018 = null;

                // Setup the Sync Output trigger to come out on M9311A Trig2 and PXI-TRIG2
                Vsg.Triggers.ExternalTrigger.Enabled = false;
                AgM938xTriggerEnum[] triggerList = new AgM938xTriggerEnum[] { AgM938xTriggerEnum.AgM938xTriggerFrontPanelTrigger2, AgM938xTriggerEnum.AgM938xTriggerPXITrigger2 };
                Vsg.Triggers2.SynchronizationOutputTrigger2.SetDestinationList(triggerList);
                Vsg.Triggers.SynchronizationOutputTrigger.Enabled = true;
                Vsg.Triggers.SynchronizationOutputTrigger.PulseWidth = 50e-6;
                Vsa.Triggers.ExternalTrigger.Holdoff = 16e-9;

                // Setup the M9391A External Trigger to use the back plane trigger
                vsaTriggerSource = AgM9393TriggerEnum.AgM9393TriggerPXITrigger2;
                //vsaTriggerSource = AgM9393TriggerEnum.AgM9393TriggerFrontPanelTrigger1;
                Vsa.Triggers.ExternalTrigger.Source = vsaTriggerSource;

                // Common Setup for VSA Trigger
                configVsaTrigger("IMMEDIATE");

            }
            catch (Exception ex)
            {
                throw (ex);
            }

        }

        public void configVsaTrigger(string triggerType, double delay = 50e-6)
        {
            // Common Setup for VSA Trigger
            switch (triggerType.ToUpper())
            {
                case "IMM":
                case "IMMEDIATE":
                    Vsa.Triggers.AcquisitionTrigger.Mode = AgM9393AcquisitionTriggerModeEnum.AgM9393AcquisitionTriggerModeImmediate;
                    break;
                case "EXT":
                case "EXTERNAL":
                    Vsa.Triggers.AcquisitionTrigger.Mode = AgM9393AcquisitionTriggerModeEnum.AgM9393AcquisitionTriggerModeExternal;
                    Vsa.Triggers.AcquisitionTrigger.Delay = delay;
                    Vsa.Triggers.AcquisitionTrigger.TimeoutMode = AgM9393TriggerTimeoutModeEnum.AgM9393TriggerTimeoutModeAutoTriggerOnTimeout;
                    Vsa.Triggers.AcquisitionTrigger.Timeout = 100;
                    Vsa.Triggers.ExternalTrigger.Slope = AgM9393TriggerSlopeEnum.AgM9393TriggerSlopePositive;
                    Vsa.Triggers.ExternalTrigger.Source = vsaTriggerSource;
                    break;
                default:
                    throw new Exception("Invalid Trigger Mode:  " + triggerType);
            }
        }

        #endregion

        #region Init M90XA Applications
        public void loadXapps(string[] xAppsNames)
        {
            if (M9000 != null)
            {
                //  Start M90XA
                M90XA = new AgM90XA();
                const string initOptions = "QueryInstrStatus=True, Simulate=True, DriverSetup= ";
                M90XA.Initialize(_key, false, false, initOptions);
                M90XAresource = M90XA.HiSLIPVisaAddress;
                // Open AgVISA connection to M9300 software
                AgVisa32.viOpenDefaultRM(out rm);
                AgVisa32.viOpen(rm, M90XAresource, 0, 0, out xApp);
                AgVisa32.viSetAttribute(xApp, AgVisa32.VI_ATTR_TMO_VALUE, 10000);
                AgVisa32.viPrintf(xApp, ":SYST:ERR:VERB ON;\n");

                // Preload the required measurements
                foreach (string thisXapp in xAppsNames)
                {
                    M90XA.ActiveMode = thisXapp;
                }

                // Wait for the X-Apps to load
                int i = 0;
                do
                {
                    Delay(1000);
                    i++;
                } while (!M90XA.ActiveMode.Contains(xAppsNames[xAppsNames.Length - 1]) && i < 30);

                // Put display in default mode
                AgVisa32.viPrintf(xApp, showM90xaDisplay ? "DISP:ENAB ON\n" : "DISP:ENAB OFF\n");
                M90XA.Visible = showM90xaDisplay;
                // Stop Sweeping
                AgVisa32.viPrintf(xApp, ":SYSTem:KLOCk 1;\n");
                AgVisa32.viPrintf(xApp, "INIT:CONT 0;:ABORT;*OPC?\n");
                string s;
                AgVisa32.viRead(xApp, out s, 800);

                // Check out the Resource to allow direct use of "M9391"
                // Checkout the sharable IVI driver session
                if (M9000.Checkout(_key, out m9000Resource) != ResourceManagerResult.Ok)
                {
                    Exception ex = new Exception("Could not lock resource plugin");
                    throw (ex);
                }
                // Cast it to the type you asked for
                Vsa = (IAgM9393)m9000Resource.Item;
                Vsa.Utility.ResetWithDefaults();
                // Adjust PLL for best performance for WLAN, GSM and EDGE
                //Vsa.Modules.Synthesizer.PLLMode = AgM9393SynthesizerPLLModeEnum.AgM9393SynthesizerPLLModeBestWideOffset;
            }
        }

        //
        // setXAppMode
        // Enables front panel control of the X-App.  If the display is turned off, it will be turned on when enabled
        // and turned back off when returning to remote control
        public void setXAppMode(bool isLocalMode)
        {
            if (M9000 != null)
            {
                if (isLocalMode)
                {

                    unlockVsa();
                    M90XA.Visible = true;
                    AgVisa32.viPrintf(xApp, ":SYSTem:KLOCk 0;\n");
                    AgVisa32.viPrintf(xApp, "DISP:ENAB ON\n");
                    AgVisa32.viPrintf(xApp, "INIT:CONT 1\n");
                }
                else
                {
                    AgVisa32.viPrintf(xApp, showM90xaDisplay ? "DISP:ENAB ON\n" : "DISP:ENAB OFF\n");
                    M90XA.Visible = showM90xaDisplay;
                    AgVisa32.viPrintf(xApp, "INIT:CONT 0\n");
                    lockVsa();
                    // Reset the M9391 Trigger
                    Vsa.Triggers.AcquisitionTrigger.Configure(AgM9393AcquisitionTriggerModeEnum.AgM9393AcquisitionTriggerModeImmediate,
                        100e-6, 0, AgM9393TriggerTimeoutModeEnum.AgM9393TriggerTimeoutModeAutoTriggerOnTimeout,
                        200, AgM9393TriggerSlopeEnum.AgM9393TriggerSlopePositive, 0, 0);
                }
            }
        }

        public void setXAppVisible(bool isVisible)
        {
            showM90xaDisplay = isVisible;
            if (M9000 != null)
            {
                M90XA.Visible = showM90xaDisplay;
                AgVisa32.viPrintf(xApp, showM90xaDisplay ? "DISP:ENAB ON\n" : "DISP:ENAB OFF\n");
            }
        }

        #endregion

    }
}
